"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SelfManagedKafkaEventSource = exports.ManagedKafkaEventSource = exports.AuthenticationMethod = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const crypto = require("crypto");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const core_1 = require("@aws-cdk/core");
const stream_1 = require("./stream");
// keep this import separate from other imports to reduce chance for merge conflicts with v2-main
// eslint-disable-next-line no-duplicate-imports, import/order
const core_2 = require("@aws-cdk/core");
/**
 * The authentication method to use with SelfManagedKafkaEventSource.
 *
 * @stability stable
 */
var AuthenticationMethod;
(function (AuthenticationMethod) {
    AuthenticationMethod["SASL_SCRAM_512_AUTH"] = "SASL_SCRAM_512_AUTH";
    AuthenticationMethod["SASL_SCRAM_256_AUTH"] = "SASL_SCRAM_256_AUTH";
    AuthenticationMethod["BASIC_AUTH"] = "BASIC_AUTH";
    AuthenticationMethod["CLIENT_CERTIFICATE_TLS_AUTH"] = "CLIENT_CERTIFICATE_TLS_AUTH";
})(AuthenticationMethod = exports.AuthenticationMethod || (exports.AuthenticationMethod = {}));
/**
 * Use a MSK cluster as a streaming source for AWS Lambda.
 *
 * @stability stable
 */
class ManagedKafkaEventSource extends stream_1.StreamEventSource {
    /**
     * @stability stable
     */
    constructor(props) {
        super(props);
        this._eventSourceMappingId = undefined;
        jsiiDeprecationWarnings._aws_cdk_aws_lambda_event_sources_ManagedKafkaEventSourceProps(props);
        this.innerProps = props;
    }
    /**
     * Called by `lambda.addEventSource` to allow the event source to bind to this function.
     *
     * @stability stable
     */
    bind(target) {
        var _c;
        const eventSourceMapping = target.addEventSourceMapping(`KafkaEventSource:${core_1.Names.nodeUniqueId(target.node)}${this.innerProps.topic}`, this.enrichMappingOptions({
            eventSourceArn: this.innerProps.clusterArn,
            startingPosition: this.innerProps.startingPosition,
            sourceAccessConfigurations: this.sourceAccessConfigurations(),
            kafkaTopic: this.innerProps.topic,
        }));
        this._eventSourceMappingId = eventSourceMapping.eventSourceMappingId;
        if (this.innerProps.secret !== undefined) {
            this.innerProps.secret.grantRead(target);
        }
        target.addToRolePolicy(new iam.PolicyStatement({
            actions: ['kafka:DescribeCluster', 'kafka:GetBootstrapBrokers', 'kafka:ListScramSecrets'],
            resources: [this.innerProps.clusterArn],
        }));
        (_c = target.role) === null || _c === void 0 ? void 0 : _c.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaMSKExecutionRole'));
    }
    sourceAccessConfigurations() {
        const sourceAccessConfigurations = [];
        if (this.innerProps.secret !== undefined) {
            // "Amazon MSK only supports SCRAM-SHA-512 authentication." from https://docs.aws.amazon.com/msk/latest/developerguide/msk-password.html#msk-password-limitations
            sourceAccessConfigurations.push({
                type: lambda.SourceAccessConfigurationType.SASL_SCRAM_512_AUTH,
                uri: this.innerProps.secret.secretArn,
            });
        }
        return sourceAccessConfigurations.length === 0
            ? undefined
            : sourceAccessConfigurations;
    }
    /**
     * The identifier for this EventSourceMapping.
     *
     * @stability stable
     */
    get eventSourceMappingId() {
        if (!this._eventSourceMappingId) {
            throw new Error('KafkaEventSource is not yet bound to an event source mapping');
        }
        return this._eventSourceMappingId;
    }
}
exports.ManagedKafkaEventSource = ManagedKafkaEventSource;
_a = JSII_RTTI_SYMBOL_1;
ManagedKafkaEventSource[_a] = { fqn: "@aws-cdk/aws-lambda-event-sources.ManagedKafkaEventSource", version: "1.144.0" };
/**
 * Use a self hosted Kafka installation as a streaming source for AWS Lambda.
 *
 * @stability stable
 */
class SelfManagedKafkaEventSource extends stream_1.StreamEventSource {
    /**
     * @stability stable
     */
    constructor(props) {
        super(props);
        jsiiDeprecationWarnings._aws_cdk_aws_lambda_event_sources_SelfManagedKafkaEventSourceProps(props);
        if (props.vpc) {
            if (!props.securityGroup) {
                throw new Error('securityGroup must be set when providing vpc');
            }
            if (!props.vpcSubnets) {
                throw new Error('vpcSubnets must be set when providing vpc');
            }
        }
        else if (!props.secret) {
            throw new Error('secret must be set if Kafka brokers accessed over Internet');
        }
        this.innerProps = props;
    }
    /**
     * Called by `lambda.addEventSource` to allow the event source to bind to this function.
     *
     * @stability stable
     */
    bind(target) {
        if (!core_2.Construct.isConstruct(target)) {
            throw new Error('Function is not a construct. Unexpected error.');
        }
        target.addEventSourceMapping(this.mappingId(target), this.enrichMappingOptions({
            kafkaBootstrapServers: this.innerProps.bootstrapServers,
            kafkaTopic: this.innerProps.topic,
            startingPosition: this.innerProps.startingPosition,
            sourceAccessConfigurations: this.sourceAccessConfigurations(),
        }));
        if (this.innerProps.secret !== undefined) {
            this.innerProps.secret.grantRead(target);
        }
    }
    mappingId(target) {
        let hash = crypto.createHash('md5');
        hash.update(JSON.stringify(core_1.Stack.of(target).resolve(this.innerProps.bootstrapServers)));
        const idHash = hash.digest('hex');
        return `KafkaEventSource:${idHash}:${this.innerProps.topic}`;
    }
    sourceAccessConfigurations() {
        var _c;
        let authType;
        switch (this.innerProps.authenticationMethod) {
            case AuthenticationMethod.BASIC_AUTH:
                authType = lambda.SourceAccessConfigurationType.BASIC_AUTH;
                break;
            case AuthenticationMethod.CLIENT_CERTIFICATE_TLS_AUTH:
                authType = lambda.SourceAccessConfigurationType.CLIENT_CERTIFICATE_TLS_AUTH;
                break;
            case AuthenticationMethod.SASL_SCRAM_256_AUTH:
                authType = lambda.SourceAccessConfigurationType.SASL_SCRAM_256_AUTH;
                break;
            case AuthenticationMethod.SASL_SCRAM_512_AUTH:
            default:
                authType = lambda.SourceAccessConfigurationType.SASL_SCRAM_512_AUTH;
                break;
        }
        const sourceAccessConfigurations = [];
        if (this.innerProps.secret !== undefined) {
            sourceAccessConfigurations.push({ type: authType, uri: this.innerProps.secret.secretArn });
        }
        if (this.innerProps.vpcSubnets !== undefined && this.innerProps.securityGroup !== undefined) {
            sourceAccessConfigurations.push({
                type: lambda.SourceAccessConfigurationType.VPC_SECURITY_GROUP,
                uri: this.innerProps.securityGroup.securityGroupId,
            });
            (_c = this.innerProps.vpc) === null || _c === void 0 ? void 0 : _c.selectSubnets(this.innerProps.vpcSubnets).subnetIds.forEach((id) => {
                sourceAccessConfigurations.push({ type: lambda.SourceAccessConfigurationType.VPC_SUBNET, uri: id });
            });
        }
        return sourceAccessConfigurations.length === 0
            ? undefined
            : sourceAccessConfigurations;
    }
}
exports.SelfManagedKafkaEventSource = SelfManagedKafkaEventSource;
_b = JSII_RTTI_SYMBOL_1;
SelfManagedKafkaEventSource[_b] = { fqn: "@aws-cdk/aws-lambda-event-sources.SelfManagedKafkaEventSource", version: "1.144.0" };
//# sourceMappingURL=data:application/json;base64,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