"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AutoScaler = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_autoscaling_1 = require("@aws-cdk/aws-autoscaling");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_elasticloadbalancingv2_1 = require("@aws-cdk/aws-elasticloadbalancingv2");
const iam = require("@aws-cdk/aws-iam");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
/**
 * @stability stable
 */
class AutoScaler extends cdk.Resource {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        const launchTemplate = this.getLT(props.templateProps, props.asgName);
        const tgArn = this.getTG(props.tgProps, props.templateProps.vpc.vpcName);
        new aws_autoscaling_1.CfnAutoScalingGroup(this, props.asgName, {
            maxSize: props.maxSize,
            minSize: props.minSize,
            autoScalingGroupName: props.asgName,
            launchTemplate: {
                version: launchTemplate.versionNumber,
                launchTemplateId: launchTemplate.launchTemplateId,
                launchTemplateName: launchTemplate.launchTemplateName,
            },
            targetGroupArns: tgArn,
            tags: props.tags,
            availabilityZones: this.getZones(props.subnets),
            vpcZoneIdentifier: props.subnets,
            healthCheckGracePeriod: 300,
        });
        if (tgArn.length) {
            this.targetGroupArn = tgArn[0];
        }
        this.targetGroupArn = '';
    }
    getVPC(props) {
        const stackVPC = aws_ec2_1.Vpc.fromLookup(this, props.vpcName, {
            isDefault: false,
            vpcId: props.vpcName,
        });
        return stackVPC;
    }
    getRole(props, asgName) {
        var _b;
        if (props.type == 'existing') {
            const role = iam.Role.fromRoleArn(this, asgName + '-stackRole', props.roleArn);
            return role;
        }
        else {
            const role = new iam.Role(this, asgName + '-stackRole', {
                assumedBy: new aws_iam_1.ServicePrincipal('ec2.amazonaws.com'),
                roleName: asgName + '-role',
            });
            role.addManagedPolicy({
                managedPolicyArn: 'arn:aws:iam::aws:policy/service-role/AmazonEC2SpotFleetTaggingRole',
            });
            role.addManagedPolicy({
                managedPolicyArn: 'arn:aws:iam::aws:policy/ReadOnlyAccess',
            });
            role.addManagedPolicy({
                managedPolicyArn: 'arn:aws:iam::aws:policy/service-role/AmazonEC2RoleforSSM',
            });
            role.addManagedPolicy({
                managedPolicyArn: 'arn:aws:iam::aws:policy/AmazonEC2FullAccess',
            });
            role.addToPolicy(aws_iam_1.PolicyStatement.fromJson({
                Version: '2012-10-17',
                Statement: [
                    {
                        Effect: 'Allow',
                        Action: [
                            'iam:ListUsers',
                            'iam:GetGroup',
                        ],
                        Resource: '*',
                    },
                    {
                        Effect: 'Allow',
                        Action: 'ec2:DescribeTags',
                        Resource: '*',
                    },
                ],
            }));
            role.addToPolicy(aws_iam_1.PolicyStatement.fromJson({
                Version: '2012-10-17',
                Statement: [
                    {
                        Sid: 'VisualEditor0',
                        Effect: 'Allow',
                        Action: [
                            'ec2:DescribeInstances',
                            'ec2:DescribeTags',
                        ],
                        Resource: '*',
                    },
                ],
            }));
            (_b = props.additionalPolicies) === null || _b === void 0 ? void 0 : _b.forEach(policyDoc => {
                role.addToPolicy(aws_iam_1.PolicyStatement.fromJson(policyDoc));
            });
            return role;
        }
    }
    getSG(props, vpc, asgName) {
        var _b, _c, _d;
        if (props.type == 'existing') {
            const securityGroup = aws_ec2_1.SecurityGroup.fromSecurityGroupId(this, 'stack-sg', props.sgGroupId);
            return securityGroup;
        }
        else {
            const sgProps = {
                securityGroupName: (_b = props.securityGroupName) !== null && _b !== void 0 ? _b : 'stack-sg-group',
                vpc: vpc,
                allowAllOutbound: (_c = props.allowAllOutbound) !== null && _c !== void 0 ? _c : true,
                disableInlineRules: (_d = props.disableInlineRules) !== null && _d !== void 0 ? _d : false,
            };
            const securityGroup = new aws_ec2_1.SecurityGroup(this, asgName + '-stack-sg', sgProps);
            props.ingressRules.forEach(ingress => {
                var _b;
                const ingressSG = aws_ec2_1.SecurityGroup.fromSecurityGroupId(this, ingress.sourceSG + ingress.port.toString(), ingress.sourceSG);
                securityGroup.connections.allowFrom(ingressSG, aws_ec2_1.Port.tcp(ingress.port), (_b = ingress.description) !== null && _b !== void 0 ? _b : 'Application port');
            });
            return securityGroup;
        }
    }
    getBD(props) {
        const bd = {
            deviceName: props.name,
            volume: aws_ec2_1.BlockDeviceVolume.ebs(props.size, {
                volumeType: props.type,
                deleteOnTermination: true,
            }),
        };
        return bd;
    }
    getLT(props, asgName) {
        if (props.type == 'existing') {
            const launchTemplate = aws_ec2_1.LaunchTemplate.fromLaunchTemplateAttributes(this, props.templateName, props.existingAttributes);
            return launchTemplate;
        }
        else {
            const launchTemplate = new aws_ec2_1.LaunchTemplate(this, props.templateName, {
                launchTemplateName: props.templateName,
                instanceType: new aws_ec2_1.InstanceType(props.instanceType),
                machineImage: aws_ec2_1.MachineImage.lookup({
                    name: props.amiImageId,
                }),
                securityGroup: this.getSG(props.securityGroup, this.getVPC(props.vpc), asgName),
                role: this.getRole(props.role, asgName),
                detailedMonitoring: false,
                blockDevices: [this.getBD(props.blockDevice)],
                keyName: props.sshKey,
            });
            return launchTemplate;
        }
    }
    getTG(props, vpcId) {
        if (props != undefined) {
            const tg = new aws_elasticloadbalancingv2_1.CfnTargetGroup(this, props.name, {
                name: props.name,
                healthCheckEnabled: true,
                healthCheckPath: props.healthPath,
                ...((props.protocol == 'GRPC') ? { protocol: 'HTTP' } : { protocol: props.protocol }),
                ...((props.protocol == 'GRPC') ? { protocolVersion: 'GRPC' } : {}),
                healthCheckTimeoutSeconds: props.timeout,
                healthCheckPort: String(props.port),
                port: props.port,
                vpcId: vpcId,
            });
            return [tg.ref];
        }
        else {
            return [];
        }
    }
    getZones(subnets) {
        var availabilityZones = [];
        subnets.forEach(subnet => {
            const net = aws_ec2_1.Subnet.fromSubnetAttributes(this, subnet, {
                availabilityZone: 'dummy',
                subnetId: subnet,
            });
            availabilityZones.push(net.availabilityZone);
        });
        return availabilityZones;
    }
}
exports.AutoScaler = AutoScaler;
_a = JSII_RTTI_SYMBOL_1;
AutoScaler[_a] = { fqn: "@smallcase/aws-cdk-microservice.AutoScaler", version: "0.0.5" };
//# sourceMappingURL=data:application/json;base64,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