"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SqsSubscription = void 0;
const iam = require("@aws-cdk/aws-iam");
const sns = require("@aws-cdk/aws-sns");
const core_1 = require("@aws-cdk/core");
/**
 * Use an SQS queue as a subscription target.
 */
class SqsSubscription {
    /**
     *
     */
    constructor(queue, props = {}) {
        this.queue = queue;
        this.props = props;
    }
    /**
     * Returns a configuration for an SQS queue to subscribe to an SNS topic.
     */
    bind(topic) {
        // Create subscription under *consuming* construct to make sure it ends up
        // in the correct stack in cases of cross-stack subscriptions.
        if (!core_1.Construct.isConstruct(this.queue)) {
            throw new Error('The supplied Queue object must be an instance of Construct');
        }
        // add a statement to the queue resource policy which allows this topic
        // to send messages to the queue.
        this.queue.addToResourcePolicy(new iam.PolicyStatement({
            resources: [this.queue.queueArn],
            actions: ['sqs:SendMessage'],
            principals: [new iam.ServicePrincipal('sns.amazonaws.com')],
            conditions: {
                ArnEquals: { 'aws:SourceArn': topic.topicArn },
            },
        }));
        return {
            subscriberScope: this.queue,
            subscriberId: topic.node.uniqueId,
            endpoint: this.queue.queueArn,
            protocol: sns.SubscriptionProtocol.SQS,
            rawMessageDelivery: this.props.rawMessageDelivery,
            filterPolicy: this.props.filterPolicy,
            region: this.regionFromArn(topic),
            deadLetterQueue: this.props.deadLetterQueue,
        };
    }
    regionFromArn(topic) {
        // no need to specify `region` for topics defined within the same stack
        if (topic instanceof sns.Topic) {
            return undefined;
        }
        return core_1.Stack.of(topic).parseArn(topic.topicArn).region;
    }
}
exports.SqsSubscription = SqsSubscription;
//# sourceMappingURL=data:application/json;base64,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