"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const route53 = require("@aws-cdk/aws-route53");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
test('apex domain selection by default', () => {
    const stack = new core_1.Stack();
    new lib_1.Certificate(stack, 'Certificate', {
        domainName: 'test.example.com',
    });
    expect(stack).toHaveResource('AWS::CertificateManager::Certificate', {
        DomainName: 'test.example.com',
        DomainValidationOptions: [{
                DomainName: 'test.example.com',
                ValidationDomain: 'example.com',
            }],
    });
});
test('validation domain can be overridden', () => {
    const stack = new core_1.Stack();
    new lib_1.Certificate(stack, 'Certificate', {
        domainName: 'test.example.com',
        validationDomains: {
            'test.example.com': 'test.example.com',
        },
    });
    expect(stack).toHaveResource('AWS::CertificateManager::Certificate', {
        DomainValidationOptions: [{
                DomainName: 'test.example.com',
                ValidationDomain: 'test.example.com',
            }],
    });
});
test('export and import', () => {
    // GIVEN
    const stack = new core_1.Stack();
    // WHEN
    const c = lib_1.Certificate.fromCertificateArn(stack, 'Imported', 'cert-arn');
    // THEN
    expect(c.certificateArn).toBe('cert-arn');
});
test('can configure validation method', () => {
    const stack = new core_1.Stack();
    new lib_1.Certificate(stack, 'Certificate', {
        domainName: 'test.example.com',
        validationMethod: lib_1.ValidationMethod.DNS,
    });
    expect(stack).toHaveResource('AWS::CertificateManager::Certificate', {
        DomainName: 'test.example.com',
        ValidationMethod: 'DNS',
    });
});
test('needs validation domain supplied if domain contains a token', () => {
    const stack = new core_1.Stack();
    expect(() => {
        const domainName = core_1.Lazy.stringValue({ produce: () => 'example.com' });
        new lib_1.Certificate(stack, 'Certificate', {
            domainName,
        });
    }).toThrow(/'validationDomains' needs to be supplied/);
});
test('validationdomains can be given for a Token', () => {
    const stack = new core_1.Stack();
    const domainName = core_1.Lazy.stringValue({ produce: () => 'my.example.com' });
    new lib_1.Certificate(stack, 'Certificate', {
        domainName,
        validationDomains: {
            [domainName]: 'example.com',
        },
    });
    expect(stack).toHaveResource('AWS::CertificateManager::Certificate', {
        DomainName: 'my.example.com',
        DomainValidationOptions: [{
                DomainName: 'my.example.com',
                ValidationDomain: 'example.com',
            }],
    });
});
test('CertificateValidation.fromEmail', () => {
    const stack = new core_1.Stack();
    new lib_1.Certificate(stack, 'Certificate', {
        domainName: 'test.example.com',
        subjectAlternativeNames: ['extra.example.com'],
        validation: lib_1.CertificateValidation.fromEmail({
            'test.example.com': 'example.com',
        }),
    });
    expect(stack).toHaveResource('AWS::CertificateManager::Certificate', {
        DomainName: 'test.example.com',
        SubjectAlternativeNames: ['extra.example.com'],
        DomainValidationOptions: [
            {
                DomainName: 'test.example.com',
                ValidationDomain: 'example.com',
            },
            {
                DomainName: 'extra.example.com',
                ValidationDomain: 'example.com',
            },
        ],
        ValidationMethod: 'EMAIL',
    });
});
describe('CertificateValidation.fromDns', () => {
    test('without a hosted zone', () => {
        const stack = new core_1.Stack();
        new lib_1.Certificate(stack, 'Certificate', {
            domainName: 'test.example.com',
            subjectAlternativeNames: ['extra.example.com'],
            validation: lib_1.CertificateValidation.fromDns(),
        });
        expect(stack).toHaveResource('AWS::CertificateManager::Certificate', {
            DomainName: 'test.example.com',
            SubjectAlternativeNames: ['extra.example.com'],
            ValidationMethod: 'DNS',
        });
    });
    test('with a hosted zone', () => {
        const stack = new core_1.Stack();
        const exampleCom = new route53.HostedZone(stack, 'ExampleCom', {
            zoneName: 'example.com',
        });
        new lib_1.Certificate(stack, 'Certificate', {
            domainName: 'test.example.com',
            validation: lib_1.CertificateValidation.fromDns(exampleCom),
        });
        expect(stack).toHaveResource('AWS::CertificateManager::Certificate', {
            DomainName: 'test.example.com',
            DomainValidationOptions: [
                {
                    DomainName: 'test.example.com',
                    HostedZoneId: {
                        Ref: 'ExampleCom20E1324B',
                    },
                },
            ],
            ValidationMethod: 'DNS',
        });
    });
    test('with hosted zone and a wildcard name', () => {
        const stack = new core_1.Stack();
        const exampleCom = new route53.HostedZone(stack, 'ExampleCom', {
            zoneName: 'example.com',
        });
        new lib_1.Certificate(stack, 'Certificate', {
            domainName: 'test.example.com',
            validation: lib_1.CertificateValidation.fromDns(exampleCom),
            subjectAlternativeNames: ['*.test.example.com'],
        });
        //Wildcard domain names are de-duped.
        expect(stack).toHaveResource('AWS::CertificateManager::Certificate', {
            DomainName: 'test.example.com',
            DomainValidationOptions: [
                {
                    DomainName: 'test.example.com',
                    HostedZoneId: {
                        Ref: 'ExampleCom20E1324B',
                    },
                },
            ],
            ValidationMethod: 'DNS',
        });
    });
    test('with hosted zone and multiple wildcard names', () => {
        const stack = new core_1.Stack();
        const exampleCom = new route53.HostedZone(stack, 'ExampleCom', {
            zoneName: 'example.com',
        });
        new lib_1.Certificate(stack, 'Certificate', {
            domainName: 'test.example.com',
            validation: lib_1.CertificateValidation.fromDns(exampleCom),
            subjectAlternativeNames: ['*.test.example.com', '*.foo.test.example.com', 'bar.test.example.com'],
        });
        //Wildcard domain names are de-duped.
        expect(stack).toHaveResource('AWS::CertificateManager::Certificate', {
            DomainName: 'test.example.com',
            DomainValidationOptions: [
                {
                    DomainName: 'test.example.com',
                    HostedZoneId: {
                        Ref: 'ExampleCom20E1324B',
                    },
                },
                {
                    DomainName: '*.foo.test.example.com',
                    HostedZoneId: {
                        Ref: 'ExampleCom20E1324B',
                    },
                },
                {
                    DomainName: 'bar.test.example.com',
                    HostedZoneId: {
                        Ref: 'ExampleCom20E1324B',
                    },
                },
            ],
            ValidationMethod: 'DNS',
        });
    });
});
test('CertificateValidation.fromDnsMultiZone', () => {
    const stack = new core_1.Stack();
    const exampleCom = new route53.HostedZone(stack, 'ExampleCom', {
        zoneName: 'example.com',
    });
    const exampleNet = new route53.HostedZone(stack, 'ExampleNet', {
        zoneName: 'example.com',
    });
    new lib_1.Certificate(stack, 'Certificate', {
        domainName: 'test.example.com',
        subjectAlternativeNames: ['cool.example.com', 'test.example.net'],
        validation: lib_1.CertificateValidation.fromDnsMultiZone({
            'test.example.com': exampleCom,
            'cool.example.com': exampleCom,
            'test.example.net': exampleNet,
        }),
    });
    expect(stack).toHaveResource('AWS::CertificateManager::Certificate', {
        DomainName: 'test.example.com',
        DomainValidationOptions: [
            {
                DomainName: 'test.example.com',
                HostedZoneId: {
                    Ref: 'ExampleCom20E1324B',
                },
            },
            {
                DomainName: 'cool.example.com',
                HostedZoneId: {
                    Ref: 'ExampleCom20E1324B',
                },
            },
            {
                DomainName: 'test.example.net',
                HostedZoneId: {
                    Ref: 'ExampleNetF7CA40C9',
                },
            },
        ],
        ValidationMethod: 'DNS',
    });
});
//# sourceMappingURL=data:application/json;base64,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