"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const fs = require("fs");
const path = require("path");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const core_1 = require("@aws-cdk/core");
const package_json_1 = require("delay/package.json");
const bundling_1 = require("../lib/bundling");
const util = require("../lib/util");
jest.mock('@aws-cdk/aws-lambda');
const writeFileSyncMock = jest.spyOn(fs, 'writeFileSync').mockReturnValue();
const existsSyncOriginal = fs.existsSync;
const existsSyncMock = jest.spyOn(fs, 'existsSync');
const originalFindUp = util.findUp;
const findUpMock = jest.spyOn(util, 'findUp').mockImplementation((name, directory) => {
    if (name === 'package.json') {
        return path.join(__dirname, '..');
    }
    return originalFindUp(name, directory);
});
const fromAssetMock = jest.spyOn(core_1.BundlingDockerImage, 'fromAsset');
beforeEach(() => {
    jest.clearAllMocks();
});
test('Parcel bundling', () => {
    bundling_1.Bundling.parcel({
        entry: '/project/folder/entry.ts',
        runtime: aws_lambda_1.Runtime.NODEJS_12_X,
        cacheDir: '/cache-dir',
        projectRoot: '/project',
        parcelEnvironment: {
            KEY: 'value',
        },
    });
    // Correctly bundles with parcel
    expect(aws_lambda_1.Code.fromAsset).toHaveBeenCalledWith('/project', {
        assetHashType: core_1.AssetHashType.BUNDLE,
        bundling: expect.objectContaining({
            environment: {
                KEY: 'value',
            },
            volumes: [{ containerPath: '/parcel-cache', hostPath: '/cache-dir' }],
            workingDirectory: '/asset-input/folder',
            command: [
                'bash', '-c',
                '$(node -p "require.resolve(\'parcel\')") build /asset-input/folder/entry.ts --target cdk-lambda --dist-dir /asset-output --no-autoinstall --no-scope-hoist --cache-dir /parcel-cache && mv /asset-output/entry.js /asset-output/index.js',
            ],
        }),
    });
    // Correctly updates package.json
    const call = writeFileSyncMock.mock.calls[0];
    expect(call[0]).toMatch('package.json');
    expect(JSON.parse(call[1])).toEqual(expect.objectContaining({
        targets: {
            'cdk-lambda': {
                context: 'node',
                includeNodeModules: {
                    'aws-sdk': false,
                },
                sourceMap: false,
                minify: false,
                engines: {
                    node: '>= 12',
                },
            },
        },
    }));
    // Searches for the package.json starting in the directory of the entry file
    expect(findUpMock).toHaveBeenCalledWith('package.json', '/project/folder');
});
test('Parcel with Windows paths', () => {
    bundling_1.Bundling.parcel({
        entry: 'C:\\my-project\\lib\\entry.ts',
        runtime: aws_lambda_1.Runtime.NODEJS_12_X,
        cacheDir: '/cache-dir',
        projectRoot: 'C:\\my-project',
    });
    expect(aws_lambda_1.Code.fromAsset).toHaveBeenCalledWith('C:\\my-project', expect.objectContaining({
        bundling: expect.objectContaining({
            command: expect.arrayContaining([
                expect.stringContaining('/lib/entry.ts'),
            ]),
        }),
    }));
});
test('Parcel bundling with externals and dependencies', () => {
    bundling_1.Bundling.parcel({
        entry: '/project/folder/entry.ts',
        runtime: aws_lambda_1.Runtime.NODEJS_12_X,
        projectRoot: '/project',
        externalModules: ['abc'],
        nodeModules: ['delay'],
    });
    // Correctly bundles with parcel
    expect(aws_lambda_1.Code.fromAsset).toHaveBeenCalledWith('/project', {
        assetHashType: core_1.AssetHashType.BUNDLE,
        bundling: expect.objectContaining({
            command: [
                'bash', '-c',
                '$(node -p "require.resolve(\'parcel\')") build /asset-input/folder/entry.ts --target cdk-lambda --dist-dir /asset-output --no-autoinstall --no-scope-hoist && mv /asset-output/entry.js /asset-output/index.js && mv /asset-input/.package.json /asset-output/package.json && cd /asset-output && npm install',
            ],
        }),
    });
    // Correctly updates package.json
    const call = writeFileSyncMock.mock.calls[0];
    expect(call[0]).toMatch('package.json');
    expect(JSON.parse(call[1])).toEqual(expect.objectContaining({
        targets: expect.objectContaining({
            'cdk-lambda': expect.objectContaining({
                includeNodeModules: {
                    delay: false,
                    abc: false,
                },
            }),
        }),
    }));
    // Correctly writes dummy package.json
    expect(writeFileSyncMock).toHaveBeenCalledWith('/project/.package.json', JSON.stringify({
        dependencies: {
            delay: package_json_1.version,
        },
    }));
});
test('Detects yarn.lock', () => {
    existsSyncMock.mockImplementation((p) => {
        if (/yarn.lock/.test(p.toString())) {
            return true;
        }
        return existsSyncOriginal(p);
    });
    bundling_1.Bundling.parcel({
        entry: '/project/folder/entry.ts',
        runtime: aws_lambda_1.Runtime.NODEJS_12_X,
        projectRoot: '/project',
        nodeModules: ['delay'],
    });
    // Correctly bundles with parcel
    expect(aws_lambda_1.Code.fromAsset).toHaveBeenCalledWith('/project', {
        assetHashType: core_1.AssetHashType.BUNDLE,
        bundling: expect.objectContaining({
            command: expect.arrayContaining([
                expect.stringMatching(/yarn\.lock.+yarn install/),
            ]),
        }),
    });
});
test('with build args', () => {
    bundling_1.Bundling.parcel({
        entry: '/project/folder/entry.ts',
        runtime: aws_lambda_1.Runtime.NODEJS_12_X,
        projectRoot: '/project',
        buildArgs: {
            HELLO: 'WORLD',
        },
    });
    expect(fromAssetMock).toHaveBeenCalledWith(expect.stringMatching(/parcel$/), expect.objectContaining({
        buildArgs: expect.objectContaining({
            HELLO: 'WORLD',
        }),
    }));
});
//# sourceMappingURL=data:application/json;base64,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